// Below ifdef required to remove warnings for unsafe version of fopen and fscanf.
// Secure version won't work cross-platform, forcing this small hack.
#ifdef _MSC_VER
#define _CRT_SECURE_NO_WARNINGS
#endif

#include "model.h"

bool Model::load(char* modelFilename, char* textureFilename, int renderingShape)
{
	_RenderingShape = renderingShape;
	bool result;

	// Load in the model data,
	result = loadModel(modelFilename);
	if (!result)	//Check if model loaded correctly
	{
		MessageBox(NULL, (LPCWSTR)"Model failed to load", (LPCWSTR)"Error", MB_OK);
		return false;
	}

	// Load the texture for this model.
	loadTexture(textureFilename);	//Load the texture passed in for this model

	return true;
}

bool Model::loadMTL(char* modelFilename, char* mtlFilename, int renderingShape)
{
	usingMTL = true;
	_RenderingShape = renderingShape;
	bool result;
	parseMTL(mtlFilename);	//Parse the MTL file passed in

	// Load in the model data,
	result = loadModel(modelFilename);
	if (!result)	//Check if model loaded correctly
	{
		MessageBox(NULL, (LPCWSTR)"Model failed to load", (LPCWSTR)"Error", MB_OK);
		return false;
	}

	loadAllTextures();	//Load all the textures specified from the MTL file
	return true;
}

void Model::loadAllTextures()
{
	for (int i = 0; i != vertexTextures.size(); i++)
	{
		if (textures.find(vertexTextures[i]) == textures.end())	//If we've not loaded the texture associated with this vertex
		{
			textures[vertexTextures[i]] = loadTextureRet("Assets/Textures/" + vertexTextures[i]);	//Load the texture
		}
	}
}

void Model::render(Vector3 position)
{
	//Bind the texture, do initial vertex rendering code
	glBindTexture(GL_TEXTURE_2D, texture);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);

	glEnableClientState(GL_VERTEX_ARRAY);
	glEnableClientState(GL_NORMAL_ARRAY);
	glEnableClientState(GL_TEXTURE_COORD_ARRAY);

	glVertexPointer(3, GL_FLOAT, 0, &vertex.front());
	glNormalPointer(GL_FLOAT, 0, &normals.front());
	glTexCoordPointer(2, GL_FLOAT, 0, &texCoords.front());

	glPushMatrix();
		glTranslatef(position.x, position.y, position.z);	//Translate to the rendering position
		std::string previousTexture = "";

		if (usingMTL)	//Per-vertex textures are only appliciable when using MTL
		{
			previousTexture = vertexTextures[0];	//Initialize the previous texture we bound to be the first one, then bind it
			glBindTexture(GL_TEXTURE_2D, textures[vertexTextures[0]]);
			glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
			glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);
		}

		//If we're using triangles
		if (_RenderingShape == GL_TRIANGLES)
		{
			//3 sides, divide and increment by 3
			for (int i = 0; i + 2 <= vertex.size() / 3.0f; i += 3)
			{
				if (usingMTL)	//Per-vertex textures are only appliciable when using MTL
				{
					if (vertexTextures[i] != previousTexture)	//If this is a different texture, rebind it
					{
						previousTexture = vertexTextures[i];
						glBindTexture(GL_TEXTURE_2D, textures[vertexTextures[i]]);
						glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
						glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);
					}
				}
				
				//Draw vertices to make up triangle
				glBegin(GL_TRIANGLES);
					glArrayElement(i);
					glArrayElement(i + 1);
					glArrayElement(i + 2);
				glEnd();
			}
		}

		//If we're using quads
		else if (_RenderingShape == GL_QUADS)
		{
			//4 sides, divide and increment by 4
			for (int i = 0; i + 3 < vertex.size() / 4.0f; i += 4)
			{
				if (usingMTL)	//Per-vertex textures are only appliciable when using MTL
				{
					if (vertexTextures[i] != previousTexture)	//If this is a different texture, rebind it
					{
						previousTexture = vertexTextures[i];
						glBindTexture(GL_TEXTURE_2D, textures[vertexTextures[i]]);
						glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
						glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);
					}
				}

				//Draw vertices to make up quad
				glBegin(GL_QUADS);
					glArrayElement(i);
					glArrayElement(i + 1);
					glArrayElement(i + 2);
					glArrayElement(i + 3);
				glEnd();
			}
		}

	glPopMatrix();

	glDisableClientState(GL_VERTEX_ARRAY);
	glDisableClientState(GL_NORMAL_ARRAY);
	glDisableClientState(GL_TEXTURE_COORD_ARRAY);
}

// Modified from a mulit-threaded version by Mark Ropper.
bool Model::loadModel(char* filename)
{
	int mtlIndex = -1;
	std::vector<Vector3> verts;
	vector<Vector3> norms;
	vector<Vector3> texCs;
	vector<unsigned int> faces;
	vector<string> mtlStrs;
	vector<string> facesTextures;

	FILE* file = fopen(filename, "r");
	if (file == NULL)
	{
		return false;
	}
	while (true)
	{
		char lineHeader[256];

		// Read first word of the line
		int res = fscanf(file, "%s", lineHeader);
		if (res == EOF)
		{
			break; // exit loop
		}
		else // Parse
		{
			if (strcmp(lineHeader, "v") == 0) // Vertex
			{
				Vector3 vertex;
				fscanf(file, "%f %f %f\n", &vertex.x, &vertex.y, &vertex.z);
				verts.push_back(vertex);
			}

			else if (strcmp(lineHeader, "vt") == 0) // Tex Coord
			{
				Vector3 uv;
				fscanf(file, "%f %f\n", &uv.x, &uv.y);
				texCs.push_back(uv);
			}

			else if (strcmp(lineHeader, "vn") == 0) // Normal
			{
				Vector3 normal;
				fscanf(file, "%f %f %f\n", &normal.x, &normal.y, &normal.z);
				norms.push_back(normal);
			}

			else if (strcmp(lineHeader, "f") == 0) // Face
			{
				unsigned int face[9];
				int matches = fscanf(file, "%d/%d/%d %d/%d/%d %d/%d/%d\n", &face[0], &face[1], &face[2],
					&face[3], &face[4], &face[5],
					&face[6], &face[7], &face[8]);
				if (matches != 9 && matches != 12)
				{
					// Parser error, or not triangle faces or quad faces
					return false;
				}

				for (int i = 0; i < 9; i++)
				{
					faces.push_back(face[i]);	//Push back all the faces

					if (usingMTL)
					{
						facesTextures.push_back(mtls[mtlStrs[mtlIndex]]);	//Get the texture for this face from the maps
					}
				}
			}

			else if (strcmp(lineHeader, "usemtl") == 0)
			{
				char str[256];
				fscanf(file, "%s", &str);
				mtlStrs.push_back(string(str));	//Push back this mtl path
				mtlIndex++;
			}
		}
	}

	for (size_t i = 0; i < faces.size(); i += 3)
	{
		//Fill arrays
		vertex.push_back(verts[faces[i] - 1].x);
		vertex.push_back(verts[faces[i] - 1].y);
		vertex.push_back(verts[faces[i] - 1].z);

		texCoords.push_back(texCs[faces[i + 1] - 1].x);
		texCoords.push_back(texCs[faces[i + 1] - 1].y);

		normals.push_back(norms[faces[i + 2] - 1].x);
		normals.push_back(norms[faces[i + 2] - 1].y);
		normals.push_back(norms[faces[i + 2] - 1].z);

		if (usingMTL)
		{
			vertexTextures.push_back(facesTextures[i]);
		}
	}

	// Once data has been sorted clear read data (which has been copied and are not longer needed).
	verts.clear();
	norms.clear();
	texCs.clear();
	faces.clear();

	return true;
}

bool Model::parseMTL(char* filename)
{
	//Open the file
	FILE* file = fopen(filename, "r");
	if (file == NULL)
	{
		return false;
	}

	while (true)
	{
		char lineHeader[256];

		// Read first word of the line
		int res = fscanf(file, "%s", &lineHeader);

		if (res == EOF)
		{
			break; // exit loop
		}

		else // Parse
		{
			if (strcmp(lineHeader, "newmtl") == 0)	//If we've found an entry
			{
				//Read thd ident
				char materialIdent[256];
				fscanf(file, "%s", &materialIdent);

				int i = 0;
				for (int i = 0; i < 11; i++)
				{
					char line[256];
					fgets(line, 256, file);
				}
			
				//Read the texture header
				char textureLineHeader[256];
				fscanf(file, "%s", &textureLineHeader);

				if (strcmp(textureLineHeader, "map_Ka") == 0)
				{
					char materialTexture[256];
					fscanf(file, "%s", materialTexture);
					
					//Use delims to cut path down to be relative and add to map
					std::set<char> delims{ '\\' };
					std::experimental::filesystem::path p(materialTexture);
					mtls[materialIdent] = p.stem().string() + ".png";
				}
			}
		}
	}
}

GLuint Model::loadTextureRet(std::string path)
{
	GLuint texture = SOIL_load_OGL_texture
	(
		path.c_str(),
		SOIL_LOAD_AUTO,
		SOIL_CREATE_NEW_ID,
		SOIL_FLAG_MIPMAPS | SOIL_FLAG_NTSC_SAFE_RGB | SOIL_FLAG_COMPRESS_TO_DXT | SOIL_FLAG_INVERT_Y // Depending on texture file type some need inverted others don't.
	);

	//check for an error during the load process
	if (texture == 0)
	{
		printf("SOIL loading error: '%s'\n", SOIL_last_result());
	}

	return texture;
}

void Model::loadTexture(char* filename)
{
	texture = SOIL_load_OGL_texture
	(
		filename,
		SOIL_LOAD_AUTO,
		SOIL_CREATE_NEW_ID,
		SOIL_FLAG_MIPMAPS | SOIL_FLAG_NTSC_SAFE_RGB | SOIL_FLAG_COMPRESS_TO_DXT | SOIL_FLAG_INVERT_Y // Depending on texture file type some need inverted others don't.
	);

	//check for an error during the load process
	if (texture == 0)
	{
		printf("SOIL loading error: '%s'\n", SOIL_last_result());
	}
}



